import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env
from copy import deepcopy as cp

class ReacherEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    def __init__(self):
        utils.EzPickle.__init__(self)
        mujoco_env.MujocoEnv.__init__(self, "reacher.xml", 2)


    def step(self, a):
        # compute distance to real target
        vec_to_real = self.get_body_com("fingertip") - self.get_body_com("target")
        reward_dist = -np.linalg.norm(vec_to_real)

        # compute distance to adversarial target (positive such that objective of adversary is to minimize it)
        adv_target = cp(self.get_body_com("target"))
        adv_target[0] *= -1
        vec_to_adv = self.get_body_com("fingertip") - adv_target
        reward_dist_adv = np.linalg.norm(vec_to_adv)

        reward_ctrl = -np.square(a).sum()
        reward = reward_dist + reward_ctrl
        self.do_simulation(a, self.frame_skip)
        ob = self._get_obs()
        done = False
        return ob, reward, done, dict(reward_dist=reward_dist, reward_ctrl=reward_ctrl, reward_dist_adv=reward_dist_adv)

    def get_pos_vel_from_obs(self, obs):

        theta = [np.arctan2(obs[2], obs[0]), np.arctan2(obs[3], obs[1])]
        pos = np.array([*theta, obs[4], obs[5]])
        vel = np.array([obs[6], obs[7], 0 , 0])

        return pos, vel

    def render(self, mode, observation=None):

        if observation is None:
            return super().render(mode)
        else:

            # target_true_x, target_true_y = self.sim.data.qpos.flat[2:]

            pos_before, vel_before = cp(self.sim.data.qpos), cp(self.sim.data.qvel)
            qpos, qvel = self.get_pos_vel_from_obs(observation)

            # replace true target position to render correct target
            # qpos[2:] = target_true_x, target_true_y

            # set all velocities to zero
            qvel = qvel * 0

            self.set_state(qpos, qvel)
            obs = super().render(mode)
            self.set_state(pos_before, vel_before)
            return obs

    def simulate_step(self, initial_observation, action):

        pos_before, vel_before = cp(self.sim.data.qpos), cp(self.sim.data.qvel)

        qpos, qvel = self.get_pos_vel_from_obs(initial_observation)
        self.set_state(qpos, qvel)

        self.do_simulation(action, self.frame_skip)

        theta = self.sim.data.qpos.flat[:2]
        next_obs = np.concatenate(
            [
                np.cos(theta),
                np.sin(theta),
                self.sim.data.qpos.flat[2:],
                self.sim.data.qvel.flat[:2],
                self.get_body_com("fingertip") - self.get_body_com("target"),
            ]
        )


        self.set_state(pos_before, vel_before)

        return next_obs

    def viewer_setup(self):
        self.viewer.cam.trackbodyid = 0

    def reset_model(self):
        qpos = (
            self.np_random.uniform(low=-0.1, high=0.1, size=self.model.nq)
            + self.init_qpos
        )
        while True:
            self.goal = self.np_random.uniform(low=-0.2, high=0.2, size=2)
            if np.linalg.norm(self.goal) < 0.2:
                break
        qpos[-2:] = self.goal
        qvel = self.init_qvel + self.np_random.uniform(
            low=-0.005, high=0.005, size=self.model.nv
        )
        qvel[-2:] = 0
        self.set_state(qpos, qvel)
        return self._get_obs()

    def _get_obs(self):
        theta = self.sim.data.qpos.flat[:2]
        obs = np.concatenate(
            [
                np.cos(theta),
                np.sin(theta),
                self.sim.data.qpos.flat[2:],
                self.sim.data.qvel.flat[:2],
                self.get_body_com("fingertip") - self.get_body_com("target"),
            ]

        )
        return obs
